<?php
/**
 * Admin setup for the plugin
 *
 * @since 1.0
 * @function	iaffpro_add_menu_links()			Add admin page for license key
 * @function	iaffpro_register_settings			Register Settings
 * @function	iaffpro_validater_and_sanitizer()	Validate And Sanitize User Input Before Its Saved To Database
 * @function	iaffpro_get_settings()				Get settings from database
 * @function	iaffpro_admin_notices() 			All the admin notices
 */

// Exit if accessed directly
if ( ! defined('ABSPATH') ) exit; 
 
/**
 * Add admin page for license key
 *
 * Page is not shown in the admin menu and can only be accessed via the link in the plugins list (near deactivate plugin).
 * @since 	1.0
 */
function iaffpro_add_menu_links() {
	add_submenu_page ( 'options.php', __('Image Attributes Pro','auto-image-attributes-pro'), __('Image Attributes Pro','auto-image-attributes-pro'), 'manage_options', 'image-attributes-pro-activation','iaffpro_admin_interface_render'  );
}
add_action( 'admin_menu', 'iaffpro_add_menu_links' );

/**
 * Register Settings
 *
 * @since 	1.0
 */
function iaffpro_register_settings() {

	// Register Setting
	register_setting( 
		'iaffpro_settings_group', 			// Group name
		'iaffpro_settings', 				// Setting name = html form <input> name on settings form
		'iaffpro_validater_and_sanitizer'	// Input sanitizer
	);
	
	// License Section
    add_settings_section(
        'iaffpro_license_section',								// ID
        __('Image Attribute Pro Activation', 'auto-image-attributes-pro'),	// Title
        'iaffpro_license_section_callback',						// Callback Function
        'image-attributes-pro-activation'						// Page slug
    );
	
	// Registered Email
    add_settings_field(
        'iaffpro_license_email_field',				// ID
        __('Registered Email', 'auto-image-attributes-pro'),	// Title
        'iaffpro_license_email_field_callback',		// Callback function
        'image-attributes-pro-activation',			// Page slug
        'iaffpro_license_section'					// Settings Section ID
    );
	
	// License Key
    add_settings_field(
        'iaffpro_license_key_field',				// ID
        __('License Key', 'auto-image-attributes-pro'),		// Title
        'iaffpro_license_key_field_callback',		// Callback function
        'image-attributes-pro-activation',			// Page slug
        'iaffpro_license_section'					// Settings Section ID
    );
	
}
add_action( 'admin_init', 'iaffpro_register_settings' );

/**
 * Validate and sanitize user input before its saved to database
 *
 * @since 1.0
 * @since 4.4 Check if base plugin is available before displaying link to settings page.
 */
function iaffpro_validater_and_sanitizer ( $settings ) {
	
	$settings['registered_email']	= sanitize_email($settings['registered_email']);
	$settings['license_key']		= sanitize_text_field($settings['license_key']);

	/**
   * Display link to Image Attributes Pro Settings page in the admin notice after settings are saved. Provided base plugin is available.
   * Also refer function iaffpro_license_key_saved_notice()
   */
	if ( function_exists( 'iaff_auto_image_attributes' ) ) {
		$message = sprintf( __( 'License information is saved. Configure <a href="%s">Image Attributes Pro &rarr;</a>', 'auto-image-attributes-pro' ), admin_url( 'options-general.php?
		page=image-attributes-from-filename' ) );
	} else {
		$message = sprintf( __( 'License information is saved. Please install and activate <a href="%s" target="_blank">Auto Image Attributes From Filename With Bulk Updater</a> to use Image Attributes Pro.' ), 'https://wordpress.org/plugins/auto-image-attributes-from-filename-with-bulk-updater/' );
	}

	add_settings_error( 'iaffpro_settings', 'iaffpro_settings-error', $message, 'success' );
	
	return $settings;
}

/**
 * Display success message when license key is saved.
 * 
 * After moving License key page from options-general.php to (https://github.com/arunbasillal/auto-image-attributes-pro/commit/e9c1e7b720845521126b7adaac5ba3c1f3ef509a)
 * the settings error after saving license key was not displayed. This function fixes that. 
 * 
 * @since 4.3.1
 * @link https://www.forumming.com/question/15081/how-should-one-implement-add-settings-error-on-custom-menu-pages
 */
function iaffpro_license_key_saved_notice() {
	settings_errors( 'iaffpro_settings' );
}
add_action( 'admin_notices', 'iaffpro_license_key_saved_notice' );

/**
 * Get settings from database
 *
 * @since 	1.0
 * @return	Array	A merged array of default and settings saved in database. 
 */
function iaffpro_get_settings() {

	$defaults = array(); // Empty for now. Might be useful in the future.
	$settings = get_option('iaffpro_settings', $defaults);
	
	return $settings;
}

/**
 * All the admin notices
 *
 * @since 1.0
 */
function iaffpro_admin_notices() {

	// Admin notice if the base plugin isnt installed 
	if ( ! function_exists( 'iaff_auto_image_attributes' ) ) { ?>
		<div class="notice notice-warning">
			<p>
				<?php

				if ( current_user_can( 'install_plugins' ) ) {
					
					if ( file_exists( WP_PLUGIN_DIR . '/auto-image-attributes-from-filename-with-bulk-updater/iaff_image-attributes-from-filename.php' ) ) {

						// Display link to activate plugin if plugin is installed but not activated.
						printf( 
							__( '<strong>Image Attributes Pro:</strong> <a href="%s" target="_blank">Auto Image Attributes From Filename With Bulk Updater</a> is installed but not activated. To use Image Attributes Pro, <a href="%s">click here and activate the plugin &rarr;</a>', 'auto-image-attributes-pro' ),
							'https://wordpress.org/plugins/auto-image-attributes-from-filename-with-bulk-updater/',
							wp_nonce_url( self_admin_url('plugins.php?action=activate&plugin=' . urlencode( 'auto-image-attributes-from-filename-with-bulk-updater/iaff_image-attributes-from-filename.php') ), 'activate-plugin_' . 'auto-image-attributes-from-filename-with-bulk-updater/iaff_image-attributes-from-filename.php' )
						);
					} else {

						// Display link to install plugin.
						printf( 
							__( '<strong>Image Attributes Pro:</strong> Please install and activate <a href="%s" target="_blank">Auto Image Attributes From Filename With Bulk Updater</a> to use Image Attributes Pro. <a href="%s">Click here to install plugin &rarr;</a>', 'auto-image-attributes-pro' ),
							'https://wordpress.org/plugins/auto-image-attributes-from-filename-with-bulk-updater/',
							wp_nonce_url( self_admin_url( 'update.php?action=install-plugin&plugin=auto-image-attributes-from-filename-with-bulk-updater' ), 'install-plugin_auto-image-attributes-from-filename-with-bulk-updater' )
						);
					}
				}
				
				?>
			</p>
		</div>
	<?php }
	
	// Admin notice if license information is not given
	$settings = iaffpro_get_settings();
	if ( empty( $settings['registered_email'] ) || empty( $settings['license_key'] ) ) { ?>
		<div class="notice notice-success">
			<p><?php printf( __( '<strong>Thank you for choosing Image Attributes Pro.</strong> <a href="%s">Please enter license info</a> to activate the plugin and download plugin updates.', 'auto-image-attributes-pro' ),
			admin_url( 'options.php?page=image-attributes-pro-activation' ) ); ?></p>
		</div>
	<?php }
}
add_action('admin_notices', 'iaffpro_admin_notices');

/**
 * Load JS and CSS in the admin.
 *
 * @since 4.4
 */
function iaffpro_enqueue_js_and_css() {
	
	$screen = get_current_screen();

	// Load only in Media Library.
	if ( $screen->id === "upload" ) {
		wp_enqueue_script( 'iaffpro-media-library-js', plugins_url( '/js/media-library.js', __FILE__ ), array( 'jquery' ), IAFFPRO_VERSION_NUM, true );
		// Refer: https://developer.wordpress.org/plugins/javascript/enqueuing/#localize
		wp_localize_script(
			'iaffpro-media-library-js',
			'iaffpro_ml_ajax',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'iaffpro_media_library_js_nonce' ),
			)
		);	
	}
}
add_action( 'admin_enqueue_scripts', 'iaffpro_enqueue_js_and_css' );